*****************************************************************************************
*Project: Political Appointments, Bureau Chiefs, and Management Performance
*Authors: David Lewis
*Data: indepthmanagement.dta
*Data: This is a dataset of 614 federal programs and their management grades.
*Description: The intent of this batch file is to read in the data, get 
*summary statistics on the 245 bureau chiefs who manage these 614 programs
*and estimate models to assess whether career managers get systematically
*higher PART management grades than appointees (NA, PAS), and then determine
*which characteristics of careerists matter for management.
*Date: June 1, 2005.
*****************************************************************************************
*The first thing I need to do is some housekeeping and then read the data into STATA.
set memory 64000
set l 72
set level 95
set matsize 800
use indepthmanagement070906.dta

*Now I create the natural log of the budget and the natural log of bureau employment.
generate lnbudg=ln(actual) if actual>0
generate lnemp=ln(emp)

*Now the degree of politicization in the bureau SES
generate pctncses=noncareerses/(noncareerses+careerses)
generate totalpapct=((noncareerses+pas+schc)/supman)*100
summarize totalpapct
hist totalpapct
generate totsq=totalpapct*totalpapct

generate lnman=ln(supman)
generate thickness=supman/emp

*These are the commands for generating the summary statistics from program data.
*This leaves just the manager-level information.
sort mgrid
by mgrid: egen avgactual=mean(actual) if actual>0
by mgrid: egen case=seq()
by mgrid: egen programs=count(programid)
by mgrid: egen avgtotal=mean(totalweighted)
by mgrid: egen avgproggr=mean(programgrade)
by mgrid: egen avgprogp=mean(programpurpose)
by mgrid: egen avgstp=mean(strategic)
by mgrid: egen avgpm=mean(programma)
by mgrid: egen avgprogr=mean(programre)
by mgrid: egen avlnemp=mean(lnemp)
by mgrid: egen avlnbudg=mean(lnbudg)
by mgrid: egen avpol=mean(totalpapct)
by mgrid: egen avlnman=mean(lnman)
by mgrid: egen avman=mean(supman)
by mgrid: egen avemp=mean(emp)

*Now I need to create some variables that will be controls. The first variable
*indicates the year the program was graded.
generate y2002=0
replace y2002=1 if programid<1000
generate y2003=0
replace y2003=1 if programid>999&programid<2000
generate y2004=0
replace y2004=1 if programid>2000

*Here I merge this data with political variables
joinby styear using mgttvc, unmatched(master)

generate newadmin=0
replace newadmin=1 if tyear==1
tab tyear, generate(ty)
generate duration=2004-year
generate dursq=duration*duration
generate demcong=demhouse*demsen
generate unidem=demcong+dempres
generate mgt=0 if public~=.&private~=.
replace mgt=1 if publicmgt==1
replace mgt=1 if private==1

summarize

*Now I will generate indicators for programtype and department and highest education 
*level achieved
tab programtype1, generate (p)
tab departmentid2, generate (d)
tab numed, generate (n)

*Now tenure squared
generate tenuresq=tenure^2

*Now create a variable that identifies just political appointees (SES, PAS, Schedule C, and PA)
generate appointee=0 if yna~=.&ypas~=.&yca~=.
replace appointee=1 if yna==1|ypas==1|apptt=="SchC"|apptt=="PA"

*Now generate an ordinal insulation scale based upon comments from Krause.
generate insulation=0
replace insulation=1 if departmentid<16
replace insulation=2 if departmentid2>16&departmentid2<20|departmentid2==21|departmentid2==23|programid==1172
replace insulation=2 if programid==1001|programid==1157|programid==2195|programid==2196|programid==131|programid==137|programid==134|programid==1001
replace insulation=3 if departmentid2==20|programid==1000|programid==2001|programid==185|programid==1002|programid==2003|programid==186|programid==183|programid==1146|programid==1147
replace insulation=3 if programid==182|programid==2198|programid==1161|programid==1158|programid==2200|programid==1159|programid==2199|programid==2197|programid==1160|programid==181
replace insulation=3 if programid==1169|programid==2204
replace insulation=4 if departmentid2==22|programid==2008|programid==2007|programid==2009|programid==2010|programid==187|programid==2201|programid==193|programid==194|programid==1168

***************************************************************************************************************************
*These are pictures of the differences in scores by manager type.
*more
*graph bar (mean) totalweighted if appt=="CA"| appt=="NA"|appt=="PAS", over(appt, relabel(1 "Career SES" 2 "Non-Career SES" 3 "PAS Appointment")) title("Figure X. OMB Management Scores by Manager Type") ylabel(0(20)100) ytitle("PART Score")
*more
*graph bar (mean) programgrade if appt=="CA"| appt=="NA"|appt=="PAS", over(appt, relabel(1 "Career SES" 2 "Non-Career SES" 3 "PAS Appointment")) title("Figure X. OMB Management Scores by Manager Type") ytitle("Program Grade")
*more
*graph bar (mean) programpurposeraw strategicplanningraw programmanagementraw programresultsraw totalweighted if appt=="CA"| appt=="NA"|appt=="PAS", over(appttype, relabel(1 "Career SES" 2 "Non-Career SES" 3 "PAS Appointment")) legend(label(1 "Program Purpose") label(2 "Strategic Planning") label(3 "Program Management") label (4 "Program Results") label (5 "Total Score")) title("Figure X. OMB Management Scores by Manager Type") ylabel(0(20)100) ytitle("PART Score")
*more
*twoway (kdensity totalweighted if yca==1) (kdensity totalweighted if yna==1) (kdensity totalweighted if ypas==1), legend (off) ytitle (Density) xtitle (PART Score)
*more
***************************************************************************************************************************
*Now for the main regressions for the paper.  One difficulty here is how to treat managers in agencies outside the merit system.
*In these cases it is difficult to tell whether they are better classified as appointees or careerists.  I exclude them here.
*I will estimate the main model later with XS as part of the base category and put these estimates in the appendix.
drop if apptt=="XS"
*First, the regressions with appointment-type variables.

*The first regression is the most simple followed by a joint test to see whether program type improves the fit of the model.
reg totalweighted appointee lnbud p1-p7, cluster(mgrid)
test p1-p7=0
more

*In this next case I follow up the regression with a joint test to see if the department dummies are necessary.
reg totalweighted appointee lnbud p1-p7 d1-d22, cluster(mgrid)


more
*Here are three models for the table.
reg totalweighted appointee lnbud p1-p7 term com programs, cluster(mgrid)
reg totalweighted appointee  lnbud p1-p7 term com programs divided dempres, cluster(mgrid)
reg totalweighted ypas yna  lnbud p1-p7 term com programs divided dempres, cluster(mgrid)
reg totalweighted ypas yna lnbud p1-p7 term com programs divided dempres lnemp totalpapct totsq lnman, cluster(mgrid)

*Now with department indicators
reg totalweighted appointee lnbud p1-p7 d1-d22, cluster(mgrid)
reg totalweighted appointee  lnbud p1-p7 term com programs divided dempres d1-d22, cluster(mgrid)



more
*Here are supplementary analyses using the raw scores.
tobit programpurpose appointee lnbud p1-p7 term com programs, ul(100)
tobit strategic appointee lnbud p1-p7 term com programs, ll(0) ul(100)
tobit programmanage appointee lnbud p1-p7 term com programs, ul(100)
tobit programresults appointee lnbud p1-p7 term com programs, ll(0)



*I now estimate some extra models to check for the robustness of the estimates.
***First, there is a possibility that appointment status is endogenous.***
ivreg totalweighted (appointee=secterm y2002 y2003 war partch ty2-ty4 who) lnbud p1-p7 term com programs divided dempres
est store ivreg

reg totalweighted appointee lnbud p1-p7 term com programs divided dempres
hausman ivreg
more

*The Hausman test suggests that I cannot reject the null that OLS is a consistent estimator/ that there is no difference
*in the coefficients.

reg appointee secterm y2002 y2003 war partch ty2-ty4 who
predict appointee_hat

reg totalweighted appointee lnbud p1-p7 term com programs divided dempres appointee_hat
test appointee_hat

*This second test suggested by Davidson and MacKinnon leads to the same conclusion.
more

***Now I repeat those regressions with ypas and yna instead of yca.***
reg totalweighted ypas yna lnbud p1-p7 , cluster(mgrid)
reg totalweighted ypas yna lnbud p1-p7 term com programs , cluster(mgrid)
reg totalweighted ypas yna lnbud p1-p7 term com programs divided dempres , cluster(mgrid)
reg totalweighted ypas yna lnbud p1-p7 term com programs divided dempres lnemp totalpapct totsq lnman, cluster(mgrid)

more


*Now some standard regression diagnostics.
reg totalweighted appointee lnbud p1-p7 term com programs divided dempres
hettest
more
rvfplot
more
predict rstud1, rstudent
stem rstud1
summarize rstud1, detail
more
kdensity rstud1, normal
more
lvr2plot, mlabel(programid)
more

***One more just using cases where a bureau was listed***
reg totalweighted appointee lnbud p1-p7 if nobureau==0, cluster(mgrid)
reg totalweighted ypas yna lnbud p1-p7 if nobureau==0, cluster(mgrid)
reg totalweighted yca lnbud p1-p7 if nobureau==0, cluster(mgrid)

***Now I estimate a few additional models for Appendix A***

*Using an ordinal measure of insulation rather than commissions or fixed terms. Also interacting this ordinal measure with
*the appointee variable.

generate appins=appointee*insulation

reg totalweighted appointee lnbud p1-p7 insulation programs, cluster(mgrid)
reg totalweighted appointee lnbud p1-p7 insulation programs divided dempres, cluster(mgrid)
reg totalweighted appointee lnbud p1-p7 insulation programs divided dempres lnemp totalpapct totsq lnman, cluster(mgrid)
reg totalweighted appointee lnbud p1-p7 insulation appins programs, cluster(mgrid)

***Controlling for unified Democratic control during a program's creation.***
reg totalweighted appointee  lnbud p1-p7 term com programs unidem, cluster(mgrid)

***Now with categorical grade.***
oprobit programgrade  appointee lnbud p1-p7 term com programs, cluster(mgrid)
oprobit programgrade  appointee  lnbud p1-p7 term com programs divided dempres, cluster(mgrid)
oprobit programgrade  appointee  lnbud p1-p7 term com programs divided dempres lnemp totalpapct totsq lnman, cluster(mgrid)
oprobit programgrade  ypas yna lnbud p1-p7 term com programs divided dempres lnemp totalpapct totsq lnman, cluster(mgrid)


************************************************************************************************************************
*I now move to the models using biographical information.  For the tenure variable there is a possibility of endogeneity.
*To account for this I will also estimate a model with instrumental variables and use year of evaluation as the 
*exogenous regressors.  There is some question whether they are completely exogenous to PART score since PART scores
*might go up over time if people learn how to do them better.  Still, it may be the best I can do.

reg totalweighted burexp tenure tenuresq public n2-n4 othdep private who congress lnbud p1-p7, cluster(mgrid)
reg totalweighted burexp tenure tenuresq public n2-n4 othdep private who congress lnbud p1-p7 term com programs, cluster(mgrid)
reg totalweighted burexp tenure tenuresq public n2-n4 othdep private who congress lnbud p1-p7 term com programs divided dempres, cluster(mgrid)
reg totalweighted burexp tenure tenuresq public n2-n4 othdep private who congress lnbud p1-p7 term com programs divided dempres lnemp totalpapct totsq lnman, cluster(mgrid)

*Now with department indicators
reg totalweighted burexp tenure tenuresq public n2-n4 othdep private who congress lnbud p1-p7 d1-d22, cluster(mgrid)
reg totalweighted burexp tenure tenuresq public n2-n4 othdep private who congress lnbud p1-p7 term com programs divided dempres d1-d22, cluster(mgrid)


*Now interacting burexp and othdep as some way of measuring generalists v. specialists.
generate general=burexp*othdep
reg totalweighted burexp tenure tenuresq public n2-n4 othdep general private who congress lnbud p1-p7 term com programs divided dempres, cluster(mgrid)
more

*Now some regression diagnostics without clustered standard errors.
*Some fun with regression diagnostics
reg totalweighted burexp tenure tenuresq public n2-n4 othdep private who congress lnbud p1-p7 term com programs divided dempres
hettest
more
rvfplot
more
predict rstud, rstudent
stem rstud
summarize rstud, detail
more
kdensity rstud, normal
more
lvr2plot, mlabel(programid)
more

*In what follows I try and estimate models that account for the possible endogeneity of tenure.  I cannot reject the null
*that OLS produces consistent estimates.
ivreg totalweighted burexp (tenure=y2003 y2004) public lnbud p1-p7 term com programs
est store ivreg1

reg totalweighted burexp tenure public lnbud p1-p7 term com programs
hausman ivreg1
more
reg tenure y2003 y2004
predict ten_hat

reg totalweighted burexp tenure public lnbud p1-p7 term com programs ten_hat
test ten_hat
more

ivreg totalweighted burexp (tenure= y2004) public lnbud p1-p7 term com programs if y2002==0
est store ivreg2

reg totalweighted burexp tenure public lnbud p1-p7 term com programs if y2002==0
hausman ivreg2


*************************************************************************************************************************
tobit programpurposeraw burexp tenure tenuresq public n2-n4 othdep private who congress lnbud p1-p7 term com programs divided dempres, ul(100)
tobit strategicplanningraw burexp tenure tenuresq public n2-n4 othdep private who congress lnbud p1-p7 term com programs, ll(0) ul(100)
tobit programmanagementraw burexp tenure tenuresq public n2-n4 othdep private who congress lnbud p1-p7 term com programs, ul(100)
tobit programresultsraw burexp tenure tenuresq public n2-n4 othdep private who congress lnbud p1-p7 term com programs, ll(0)
oprobit programgrade burexp tenure tenuresq public n2-n4 othdep private who congress lnbud p1-p7 term com programs, cluster(mgrid)
more
*************************************************************************************************************************
*This is the way I get summary statistics.  I average across managers and by appointment type summarize.  There are some
*differences in education type across appointment authorities but not really on the careerist/appointee divide.
drop if case~=1

sort appt
by appt: summarize numed burexp othdep mgt public private tenure congress who avgtotal avgproggr avgprogp avgstp avgpm avgprogr programs avgactual avemp avpol avman

**********************************************************************************************************************

*Now regressions for Appendix B in the paper.
reg avgtotal appointee term com programs avlnbudg
reg avgtotal ypas yna term com programs avlnbudg
reg avgproggr appointee term com programs avlnbudg

generate avpolsq=avpol*avpol
*Now more fully specified but with fewer cases.
reg avgtotal appointee term com programs avlnemp avlnbudg avpol avpolsq avlnman
reg avgtotal ypas yna term com programs avlnemp avlnbudg avpol avpolsq avlnman
reg avgproggr appointee term com programs avlnemp avlnbudg avpol avpolsq avlnman
 
reg avgtotal burexp tenure tenuresq public n2-n4 othdep private who congress term com programs
reg avgtotal burexp tenure tenuresq public n2-n4 othdep private who congress term com programs avlnemp avlnbudg avpol avpolsq  avlnman



